package com.arisweb.pdfeditor;

import android.Manifest;
import android.app.Activity;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.pdf.PdfDocument;
import android.net.Uri;
import android.os.Bundle;
import android.os.Environment;
import android.provider.MediaStore;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.app.AppCompatActivity;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;
import androidx.core.content.FileProvider;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

public class MainActivity extends AppCompatActivity {

    private static final int PERMISSION_REQUEST_CODE = 100;
    private static final int IMAGE_PICK_REQUEST = 200;
    
    private EditText etTextContent;
    private EditText etFileName;
    private Button btnCreatePdf;
    private Button btnOpenPdf;
    private Button btnSavePdf;
    private Button btnSharePdf;
    private Button btnAddImage;
    private Button btnPreview;
    private TextView tvStatus;
    
    private PdfDocument currentPdfDocument;
    private File currentPdfFile;
    private Bitmap selectedImage;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);
        
        initViews();
        setupClickListeners();
        checkPermissions();
    }

    private void initViews() {
        etTextContent = findViewById(R.id.etTextContent);
        etFileName = findViewById(R.id.etFileName);
        btnCreatePdf = findViewById(R.id.btnCreatePdf);
        btnOpenPdf = findViewById(R.id.btnOpenPdf);
        btnSavePdf = findViewById(R.id.btnSavePdf);
        btnSharePdf = findViewById(R.id.btnSharePdf);
        btnAddImage = findViewById(R.id.btnAddImage);
        btnPreview = findViewById(R.id.btnPreview);
        tvStatus = findViewById(R.id.tvStatus);
        
        // Устанавливаем имя файла по умолчанию
        String defaultFileName = "PDF_" + new SimpleDateFormat("yyyyMMdd_HHmmss", Locale.getDefault()).format(new Date());
        etFileName.setText(defaultFileName);
    }

    private void setupClickListeners() {
        btnCreatePdf.setOnClickListener(v -> createPdf());
        btnOpenPdf.setOnClickListener(v -> openPdf());
        btnSavePdf.setOnClickListener(v -> savePdf());
        btnSharePdf.setOnClickListener(v -> sharePdf());
        btnAddImage.setOnClickListener(v -> selectImage());
        btnPreview.setOnClickListener(v -> previewPdf());
    }

    private void checkPermissions() {
        String[] permissions = {
            Manifest.permission.READ_EXTERNAL_STORAGE,
            Manifest.permission.WRITE_EXTERNAL_STORAGE
        };
        
        boolean allPermissionsGranted = true;
        for (String permission : permissions) {
            if (ContextCompat.checkSelfPermission(this, permission) != PackageManager.PERMISSION_GRANTED) {
                allPermissionsGranted = false;
                break;
            }
        }
        
        if (!allPermissionsGranted) {
            ActivityCompat.requestPermissions(this, permissions, PERMISSION_REQUEST_CODE);
        }
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        
        if (requestCode == PERMISSION_REQUEST_CODE) {
            boolean allGranted = true;
            for (int result : grantResults) {
                if (result != PackageManager.PERMISSION_GRANTED) {
                    allGranted = false;
                    break;
                }
            }
            
            if (!allGranted) {
                showToast(getString(R.string.permission_required));
            }
        }
    }

    private void createPdf() {
        String textContent = etTextContent.getText().toString().trim();
        String fileName = etFileName.getText().toString().trim();
        
        if (textContent.isEmpty() && selectedImage == null) {
            showToast("Добавьте текст или изображение для создания PDF");
            return;
        }
        
        if (fileName.isEmpty()) {
            showToast("Введите имя файла");
            return;
        }
        
        updateStatus(getString(R.string.status_creating));
        
        try {
            // Создаем PDF документ
            currentPdfDocument = new PdfDocument();
            PdfDocument.PageInfo pageInfo = new PdfDocument.PageInfo.Builder(595, 842, 1).create(); // A4 размер
            PdfDocument.Page page = currentPdfDocument.startPage(pageInfo);
            
            Canvas canvas = page.getCanvas();
            Paint paint = new Paint();
            paint.setTextSize(16);
            paint.setColor(android.graphics.Color.BLACK);
            
            int yPosition = 50;
            
            // Добавляем текст
            if (!textContent.isEmpty()) {
                String[] lines = textContent.split("\\n");
                for (String line : lines) {
                    canvas.drawText(line, 50, yPosition, paint);
                    yPosition += 25;
                }
            }
            
            // Добавляем изображение
            if (selectedImage != null) {
                yPosition += 20;
                int imageWidth = Math.min(selectedImage.getWidth(), 500);
                int imageHeight = (int) ((float) selectedImage.getHeight() * imageWidth / selectedImage.getWidth());
                
                Bitmap scaledImage = Bitmap.createScaledBitmap(selectedImage, imageWidth, imageHeight, true);
                canvas.drawBitmap(scaledImage, 50, yPosition, paint);
            }
            
            currentPdfDocument.finishPage(page);
            
            // Включаем кнопки сохранения и обмена
            btnSavePdf.setEnabled(true);
            btnSharePdf.setEnabled(true);
            
            updateStatus(getString(R.string.pdf_created));
            showToast(getString(R.string.pdf_created));
            
        } catch (Exception e) {
            updateStatus(getString(R.string.error_creating_pdf));
            showToast(getString(R.string.error_creating_pdf) + ": " + e.getMessage());
        }
    }

    private void savePdf() {
        if (currentPdfDocument == null) {
            showToast("Сначала создайте PDF");
            return;
        }
        
        String fileName = etFileName.getText().toString().trim();
        if (!fileName.endsWith(".pdf")) {
            fileName += ".pdf";
        }
        
        updateStatus(getString(R.string.status_saving));
        
        try {
            // Создаем папку Downloads если её нет
            File downloadsDir = Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_DOWNLOADS);
            if (!downloadsDir.exists()) {
                downloadsDir.mkdirs();
            }
            
            currentPdfFile = new File(downloadsDir, fileName);
            FileOutputStream fos = new FileOutputStream(currentPdfFile);
            currentPdfDocument.writeTo(fos);
            fos.close();
            
            updateStatus(getString(R.string.pdf_saved) + ": " + currentPdfFile.getAbsolutePath());
            showToast(getString(R.string.pdf_saved));
            
        } catch (IOException e) {
            updateStatus(getString(R.string.error_saving_pdf));
            showToast(getString(R.string.error_saving_pdf) + ": " + e.getMessage());
        }
    }

    private void sharePdf() {
        if (currentPdfFile == null) {
            showToast("Сначала сохраните PDF");
            return;
        }
        
        updateStatus(getString(R.string.status_sharing));
        
        try {
            Uri pdfUri = FileProvider.getUriForFile(this, "com.arisweb.pdfeditor.fileprovider", currentPdfFile);
            
            Intent shareIntent = new Intent(Intent.ACTION_SEND);
            shareIntent.setType("application/pdf");
            shareIntent.putExtra(Intent.EXTRA_STREAM, pdfUri);
            shareIntent.putExtra(Intent.EXTRA_SUBJECT, "PDF документ");
            shareIntent.putExtra(Intent.EXTRA_TEXT, "Отправляю PDF документ, созданный в PDF Editor");
            shareIntent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
            
            startActivity(Intent.createChooser(shareIntent, getString(R.string.dialog_share_title)));
            
            updateStatus(getString(R.string.pdf_shared));
            
        } catch (Exception e) {
            updateStatus(getString(R.string.error_sharing_pdf));
            showToast(getString(R.string.error_sharing_pdf) + ": " + e.getMessage());
        }
    }

    private void selectImage() {
        Intent intent = new Intent(Intent.ACTION_PICK, MediaStore.Images.Media.EXTERNAL_CONTENT_URI);
        startActivityForResult(intent, IMAGE_PICK_REQUEST);
    }

    private void previewPdf() {
        if (currentPdfDocument == null) {
            showToast("Сначала создайте PDF");
            return;
        }
        
        showToast("Предпросмотр: PDF готов к сохранению и отправке");
    }

    private void openPdf() {
        Intent intent = new Intent(Intent.ACTION_OPEN_DOCUMENT);
        intent.setType("application/pdf");
        intent.addCategory(Intent.CATEGORY_OPENABLE);
        startActivityForResult(intent, 300);
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, @Nullable Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        
        if (requestCode == IMAGE_PICK_REQUEST && resultCode == Activity.RESULT_OK && data != null) {
            try {
                Uri imageUri = data.getData();
                selectedImage = MediaStore.Images.Media.getBitmap(getContentResolver(), imageUri);
                showToast("Изображение добавлено");
            } catch (IOException e) {
                showToast("Ошибка при загрузке изображения: " + e.getMessage());
            }
        }
    }

    private void updateStatus(String status) {
        tvStatus.setText(status);
    }

    private void showToast(String message) {
        Toast.makeText(this, message, Toast.LENGTH_SHORT).show();
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        if (currentPdfDocument != null) {
            currentPdfDocument.close();
        }
    }
}